<?php

use Behat\Behat\Context\BehatContext;
use Behat\Behat\Context\ClosuredContextInterface;
use Behat\Behat\Event\SuiteEvent;
use WP_CLI\Process;
use WP_CLI\Utils;

// Inside a community package
if ( file_exists( __DIR__ . '/utils.php' ) ) {
	require_once __DIR__ . '/utils.php';
	require_once __DIR__ . '/Process.php';
	$project_composer = dirname( dirname( dirname( __FILE__ ) ) ) . '/composer.json';
	if ( file_exists( $project_composer ) ) {
		$composer = json_decode( file_get_contents( $project_composer ) );
		if ( ! empty( $composer->autoload->files ) ) {
			$contents = 'require:' . PHP_EOL;
			foreach ( $composer->autoload->files as $file ) {
				$contents .= '  - ' . dirname( dirname( dirname( __FILE__ ) ) ) . '/' . $file;
			}
			@mkdir( sys_get_temp_dir() . '/wp-cli-package-test/' );
			$project_config = sys_get_temp_dir() . '/wp-cli-package-test/config.yml';
			file_put_contents( $project_config, $contents );
			putenv( 'WP_CLI_CONFIG_PATH=' . $project_config );
		}
	}
// Inside WP-CLI
} else {
	require_once __DIR__ . '/../../php/utils.php';
	require_once __DIR__ . '/../../php/WP_CLI/Process.php';
	require_once __DIR__ . '/../../vendor/autoload.php';
}


/**
 * Features context.
 */
class FeatureContext extends BehatContext implements ClosuredContextInterface {

	public static  $questions     = array();
	private static $cache_dir, $suite_cache_dir;
	private static $db_settings   = array(
		'dbname' => 'wp_cli_test',
		'dbuser' => 'wp_cli_test',
		'dbpass' => 'password1',
		'dbhost' => '127.0.0.1',
	);
	public         $variables     = array();
	private        $running_procs = array();

	/**
	 * Initializes context.
	 * Every scenario gets it's own context object.
	 *
	 * @param array $parameters context parameters (set them up through behat.yml)
	 */
	public function __construct( array $parameters ) {
		$this->drop_db();
		$this->set_cache_dir();
		$this->variables['CORE_CONFIG_SETTINGS'] = Utils\assoc_args_to_str( self::$db_settings );
	}

	// We cache the results of `wp core download` to improve test performance
	// Ideally, we'd cache at the HTTP layer for more reliable tests

	public function drop_db() {
		$dbname = self::$db_settings['dbname'];
		self::run_sql( "DROP DATABASE IF EXISTS $dbname" );
	}

	private function set_cache_dir() {
		$path = sys_get_temp_dir() . '/wp-cli-test-cache';
		$this->proc( Utils\esc_cmd( 'mkdir -p %s', $path ) )->run_check();
		$this->variables['CACHE_DIR'] = $path;
	}

	private static function run_sql( $sql ) {
		Utils\run_mysql_command(
			'mysql --no-defaults', array(
				'execute' => $sql,
				'host'    => self::$db_settings['dbhost'],
				'user'    => self::$db_settings['dbuser'],
				'pass'    => self::$db_settings['dbpass'],
			)
		);
	}

	public function proc( $command, $assoc_args = array(), $path = '' ) {
		if ( ! empty( $assoc_args ) ) {
			$command .= Utils\assoc_args_to_str( $assoc_args );
		}

		$env = self::get_process_env_variables();
		if ( isset( $this->variables['SUITE_CACHE_DIR'] ) ) {
			$env['WP_CLI_CACHE_DIR'] = $this->variables['SUITE_CACHE_DIR'];
		}

		if ( isset( $this->variables['RUN_DIR'] ) ) {
			$cwd = "{$this->variables['RUN_DIR']}/{$path}";
		} else {
			$cwd = null;
		}

		return Process::create( $command, $cwd, $env );
	}

	/**
	 * Get the environment variables required for launched `wp` processes
	 *
	 * @beforeSuite
	 */
	private static function get_process_env_variables() {
		// Ensure we're using the expected `wp` binary
		$bin_dir = getenv( 'WP_CLI_BIN_DIR' ) ?: realpath( __DIR__ . "/../../bin" );
		$env     = array(
			'PATH'      => $bin_dir . ':' . getenv( 'PATH' ),
			'BEHAT_RUN' => 1,
			'HOME'      => '/tmp/wp-cli-home',
		);
		if ( $config_path = getenv( 'WP_CLI_CONFIG_PATH' ) ) {
			$env['WP_CLI_CONFIG_PATH'] = $config_path;
		}

		return $env;
	}

	/**
	 * @BeforeSuite
	 */
	public static function prepare( SuiteEvent $event ) {
		$result = Process::create( 'wp cli info', null, self::get_process_env_variables() )->run_check();
		echo PHP_EOL;
		echo $result->stdout;
		echo PHP_EOL;
		self::cache_wp_files();
	}

	private static function cache_wp_files() {
		self::$cache_dir = sys_get_temp_dir() . '/wp-cli-test core-download-cache';

		if ( is_readable( self::$cache_dir . '/wp-config-sample.php' ) ) {
			return;
		}

		$cmd = Utils\esc_cmd( 'wp core download --force --path=%s', self::$cache_dir );
		Process::create( $cmd, null, self::get_process_env_variables() )->run_check();
	}

	/**
	 * @AfterSuite
	 */
	public static function afterSuite( SuiteEvent $event ) {
		if ( self::$suite_cache_dir ) {
			Process::create( Utils\esc_cmd( 'rm -r %s', self::$suite_cache_dir ), null, self::get_process_env_variables() )->run();
		}
	}

	public static function create_cache_dir() {
		self::$suite_cache_dir = sys_get_temp_dir() . '/' . uniqid( "wp-cli-test-suite-cache-", true );
		mkdir( self::$suite_cache_dir );

		return self::$suite_cache_dir;
	}

	/**
	 * @BeforeScenario
	 */
	public function beforeScenario( $event ) {
		$this->variables['SRC_DIR'] = realpath( __DIR__ . '/../..' );
	}

	/**
	 * @AfterScenario
	 */
	public function afterScenario( $event ) {
		if ( isset( $this->variables['RUN_DIR'] ) ) {
			// remove altered WP install, unless there's an error
			if ( $event->getResult() < 4 ) {
				$this->proc( Utils\esc_cmd( 'rm -r %s', $this->variables['RUN_DIR'] ) )->run();
			}
		}

		foreach ( $this->running_procs as $proc ) {
			self::terminate_proc( $proc );
		}

		$this->cleanParameters();
		$this->cleanDataTempDir();
	}

	/**
	 * Terminate a process and any of its children.
	 */
	private static function terminate_proc( $proc ) {
		$status = proc_get_status( $proc );

		$master_pid = $status['pid'];

		$output = `ps -o ppid,pid,command | grep $master_pid`;

		foreach ( explode( PHP_EOL, $output ) as $line ) {
			if ( preg_match( '/^\s*(\d+)\s+(\d+)/', $line, $matches ) ) {
				$parent = $matches[1];
				$child  = $matches[2];

				if ( $parent == $master_pid ) {
					if ( ! posix_kill( (int) $child, 9 ) ) {
						throw new RuntimeException( posix_strerror( posix_get_last_error() ) );
					}
				}
			}
		}

		if ( ! posix_kill( (int) $master_pid, 9 ) ) {
			throw new RuntimeException( posix_strerror( posix_get_last_error() ) );
		}
	}

	public function cleanParameters() {
		$this->variables['appendParameter'] = null;
	}

	/**
	 * @BeforeScenario @cleanTemp
	 */
	public function cleanDataTempDir() {
		recursiveRmdir( $this->get_data_dir( '/temp' ) );
		mkdir( $this->get_data_dir( '/temp' ) );
	}

	public function get_data_dir( $path = '' ) {
		$dataDir = dirname( dirname( __FILE__ ) ) . '/_data';

		return ! empty( $path ) ? $dataDir . '/' . ltrim( $path, '/' ) : $dataDir;
	}

	public function getStepDefinitionResources() {
		return glob( __DIR__ . '/../steps/*.php' );
	}

	public function getHookDefinitionResources() {
		return array();
	}

	public function replace_variables( $str ) {
		return preg_replace_callback( '/\{([A-Z_]+)\}/', array( $this, '_replace_var' ), $str );
	}

	public function build_phar( $version = 'same' ) {
		$this->variables['PHAR_PATH'] = $this->variables['RUN_DIR'] . '/' . uniqid( "wp-cli-build-", true ) . '.phar';

		$this->proc(
			Utils\esc_cmd(
				'php -dphar.readonly=0 %1$s %2$s --version=%3$s && chmod +x %2$s', __DIR__ . '/../../utils/make-phar.php',
				$this->variables['PHAR_PATH'], $version
			)
		)->run_check();
	}

	/**
	 * start a background process. will automatically be closed when the tests finish.
	 */
	public function background_proc( $cmd ) {
		$descriptors = array(
			0 => STDIN,
			1 => array( 'pipe', 'w' ),
			2 => array( 'pipe', 'w' ),
		);

		$proc = proc_open( $cmd, $descriptors, $pipes, $this->variables['run_dir'], self::get_process_env_variables() );

		sleep( 1 );

		$status = proc_get_status( $proc );

		if ( ! $status['running'] ) {
			throw new runtimeexception( stream_get_contents( $pipes[2] ) );
		} else {
			$this->running_procs[] = $proc;
		}
	}

	public function move_files( $src, $dest ) {
		rename( $this->variables['RUN_DIR'] . "/$src", $this->variables['RUN_DIR'] . "/$dest" );
	}

	public function add_line_to_wp_config( &$wp_config_code, $line ) {
		$token = "/* That's all, stop editing!";

		$wp_config_code = str_replace( $token, "$line\n\n$token", $wp_config_code );
	}

	public function install_wp( $subdir = '' ) {
		$this->create_db();
		$this->create_run_dir();
		$this->download_wp( $subdir );

		$this->create_config( $subdir );

		$install_args = array(
			'url'            => 'http://example.com',
			'title'          => 'WP CLI Site',
			'admin_user'     => 'admin',
			'admin_email'    => 'admin@example.com',
			'admin_password' => 'password1'
		);

		$this->proc( 'wp core install', $install_args, $subdir )->run_check();
	}

	public function create_db() {
		$dbname = self::$db_settings['dbname'];
		self::run_sql( "CREATE DATABASE IF NOT EXISTS $dbname" );
	}

	public function create_run_dir() {
		if ( ! isset( $this->variables['RUN_DIR'] ) ) {
			$this->variables['RUN_DIR'] = sys_get_temp_dir() . '/' . uniqid( "wp-cli-test-run-", true );
			mkdir( $this->variables['RUN_DIR'] );
		}
	}

	public function download_wp( $subdir = '' ) {
		$dest_dir = $this->variables['RUN_DIR'] . "/$subdir";

		if ( $subdir ) {
			mkdir( $dest_dir );
		}

		$this->proc( Utils\esc_cmd( "cp -r %s/* %s", self::$cache_dir, $dest_dir ) )->run_check();

		// disable emailing
		mkdir( $dest_dir . '/wp-content/mu-plugins' );
		copy( __DIR__ . '/../extra/no-mail.php', $dest_dir . '/wp-content/mu-plugins/no-mail.php' );
	}

	public function create_config( $subdir = '' ) {
		$params             = self::$db_settings;
		$params['dbprefix'] = $subdir ?: 'wp_';

		$params['skip-salts'] = true;
		$this->proc( 'wp core config', $params, $subdir )->run_check();
	}

	/**
	 * @BeforeScenario @pathEnv
	 */
	public function backupPathEnvVar() {
		$this->pathBackup = getenv( 'PATH' );
		$newPath          = 'PATH=' . $this->get_data_dir() . ':' . $this->pathBackup;
		putenv( $newPath );
		$this->setupSemaphore();
	}

	/**
	 * @AfterScenario @pathEnv
	 */
	public function restorePathEnvVar() {
		putenv( 'PATH=' . $this->pathBackup );
	}

	/**
	 * @beforeStep @mockWpcept
	 */
	public function createMockWpcept() {
		if ( empty( $this->variables['cwd'] ) || ! empty( $this->variables['wpceptMocked'] ) ) {
			return;
		}

		$cwd = $this->variables['cwd'];

		$source = $this->get_data_dir( 'mockVendorStructure' );

		$this->proc( Utils\esc_cmd( "cp -r %s/* %s", $source, $cwd ) )->run_check();

		if ( ! file_exists( $cwd . '/vendor/bin/wpcept' ) ) {
			throw new RuntimeException( "Could not create mock wpcept file in '{$cwd}'" );
		}

		$this->variables['wpceptMocked'] = true;
	}

	/**
	 * @afterScenario @mockWpcept
	 */
	public function unlinkMockWpcept() {
		if ( empty( $this->variables['wpceptMocked'] ) ) {
			return;
		}

		$path = $this->variables['cwd'] . '/vendor/bin/wpcept';

		if ( file_exists( $path ) ) {
			if ( ! unlink( $path ) ) {
				throw new RuntimeException( "Could not unlink mock wpcept file at '{$path}'" );
			}
		}
	}

	/**
	 * @beforeScenario @badComposer
	 */
	public function mockBadComposer() {
		$composerFile     = $this->get_data_dir( 'composer' );
		$badComposerFile  = $this->get_data_dir( 'badComposer' );
		$tempComposerFile = $this->get_data_dir( 'tempComposer' );

		rename( $composerFile, $tempComposerFile );
		rename( $badComposerFile, $composerFile );
	}

	/**
	 * @afterScenario @badComposer
	 */
	public function restoreFakeComposer() {
		$composerFile     = $this->get_data_dir( 'composer' );
		$badComposerFile  = $this->get_data_dir( 'badComposer' );
		$tempComposerFile = $this->get_data_dir( 'tempComposer' );

		rename( $composerFile, $badComposerFile );
		rename( $tempComposerFile, $composerFile );
	}

	protected function setupSemaphore() {
		$semaphore = 100;
		$segment   = 200;

		$sem      = sem_get( $semaphore, 1, 0600 );

		$acquired = sem_acquire( $sem );

		if ( ! $acquired ) {
			throw new RuntimeException( 'Could not connect to semaphore' );
		}

		$shm       = shm_attach( $segment, 16384, 0600 );
		$processes = shm_put_var( $shm, 23, array() );
		shm_detach( $shm );
		sem_release( $sem );
	}

	private function _replace_var( $matches ) {
		$cmd = $matches[0];

		foreach ( array_slice( $matches, 1 ) as $key ) {
			$cmd = str_replace( '{' . $key . '}', $this->variables[ $key ], $cmd );
		}

		return $cmd;
	}
}
